/*
 * $Id: screen.c,v 1.6 2003/08/10 03:21:28 kenta Exp $
 *
 * Copyright 2003 Kenta Cho. All rights reserved.
 */

/**
 * OpenGL screen handler.
 *
 * @version $Revision: 1.6 $
 */
#include <stdio.h>
#include <stdlib.h>
#include <eglport.h>

#include "SDL.h"

#include <math.h>
#include <string.h>

#include "genmcr.h"
#include "screen.h"
#include "rr.h"
#include "degutil.h"
#include "attractmanager.h"
#include "letterrender.h"
#include "boss_mtd.h"

#ifdef HAVE_GLES
#define glOrtho glOrthof
#endif
#define FAR_PLANE 720

#define SCREEN_WIDTH 800
#define SCREEN_HEIGHT 480
#define LOWRES_SCREEN_WIDTH 320
#define LOWRES_SCREEN_HEIGHT 240
#define SHARE_LOC "./"

static int screenWidth, screenHeight;

// Reset viewport when the screen is resized.
static void screenResized() {
  glViewport(0, 0, 800, 480);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(45.0f, (GLfloat)screenWidth/(GLfloat)screenHeight, 0.1f, FAR_PLANE);
  glMatrixMode(GL_MODELVIEW);
}

void resized(int width, int height) {
  screenWidth = width; screenHeight = height;
  screenResized();
}

// Init OpenGL.
static void initGL() {
#ifndef HAVE_GLES
  glViewport(0, 0, screenWidth, screenHeight);
#else
  glViewport(0, 0, 800, 480);
#endif
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

  glLineWidth(1);
  glEnable(GL_LINE_SMOOTH);

  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glEnable(GL_BLEND);

  glDisable(GL_LIGHTING);
  glDisable(GL_CULL_FACE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_COLOR_MATERIAL);

  resized(screenWidth, screenHeight);
}

// Load bitmaps and convert to textures.
void loadGLTexture(char *fileName, GLuint *texture) {
  SDL_Surface *surface;

  char name[32];
  strcpy(name, SHARE_LOC);
  strcat(name, "images/");
  strcat(name, fileName);
  surface = SDL_LoadBMP(name);
  if ( !surface ) {
    fprintf(stderr, "Unable to load texture: %s\n", SDL_GetError());
#ifdef HAVE_GLES
	EGL_Destroy();
#endif
    SDL_Quit();
    exit(1);
  }

  glGenTextures(1, texture);
  glBindTexture(GL_TEXTURE_2D, *texture);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
  gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, surface->w, surface->h, GL_RGB, GL_UNSIGNED_BYTE, surface->pixels);
}

void generateTexture(GLuint *texture) {
  glGenTextures(1, texture);
}

void deleteTexture(GLuint *texture) {
  glDeleteTextures(1, texture);
}

static GLuint starTexture;
#define STAR_BMP "star.bmp"
static GLuint smokeTexture;
#define SMOKE_BMP "smoke.bmp"
static GLuint titleTexture;
#define TITLE_BMP "title.bmp"

int lowres = 1;
int windowMode = 0;
int brightness = DEFAULT_BRIGHTNESS;
Uint8 *keys;
SDL_Joystick *stick = NULL;
int joystickMode = 1;

void initSDL() {
  Uint32 videoFlags;

  if ( lowres ) {
    screenWidth  = LOWRES_SCREEN_WIDTH;
    screenHeight = LOWRES_SCREEN_HEIGHT;
  } else {
    screenWidth  = SCREEN_WIDTH;
    screenHeight = SCREEN_HEIGHT;
  }

  /* Initialize SDL */
#ifdef HAVE_GLES
        if (!EGL_Open())
                exit(1);
#endif
  if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
    fprintf(stderr, "Unable to initialize SDL: %s\n", SDL_GetError());
    exit(1);
  }
  if ( SDL_InitSubSystem(SDL_INIT_JOYSTICK) < 0 ) {
    fprintf(stderr, "Unable to initialize SDL_JOYSTICK: %s\n", SDL_GetError());
    joystickMode = 0;
  }

  /* Create an OpenGL screen */
  if ( windowMode ) {
    videoFlags = SDL_OPENGL | SDL_RESIZABLE;
  } else {
    videoFlags = SDL_OPENGL | SDL_FULLSCREEN;
  }
#ifndef HAVE_GLES
  if ( SDL_SetVideoMode(screenWidth, screenHeight, 0, videoFlags) == NULL ) {
    fprintf(stderr, "Unable to create OpenGL screen: %s\n", SDL_GetError());
    SDL_Quit();
    exit(2);
  }
#else
    SDL_SetVideoMode(800, 480, 0, SDL_SWSURFACE | SDL_FULLSCREEN ); 
    EGL_Init(0);
#endif


  if (joystickMode == 1) {
    stick = SDL_JoystickOpen(0);
  }

  /* Set the title bar in environments that support it */
  SDL_WM_SetCaption(CAPTION, NULL);

  initGL();
  loadGLTexture(STAR_BMP, &starTexture);
  loadGLTexture(SMOKE_BMP, &smokeTexture);
  loadGLTexture(TITLE_BMP, &titleTexture);

  SDL_ShowCursor(SDL_DISABLE);
}

void closeSDL() {
  SDL_ShowCursor(SDL_ENABLE);
}

float zoom = 15;
static int screenShakeCnt = 0;
static int screenShakeType = 0;

static void setEyepos() {
  float x, y;
  glPushMatrix();
  if ( screenShakeCnt > 0 ) {
    switch ( screenShakeType ) {
    case 0:
      x = (float)randNS2(256)/5000.0f;
      y = (float)randNS2(256)/5000.0f;
      break;
    default:
      x = (float)randNS2(256)*screenShakeCnt/21000.0f;
      y = (float)randNS2(256)*screenShakeCnt/21000.0f;
      break;
    }
    gluLookAt(0, 0, zoom, x, y, 0, 0.0f, 1.0f, 0.0f);
  } else {
    gluLookAt(0, 0, zoom, 0, 0, 0, 0.0f, 1.0f, 0.0f);
  }
}

void setScreenShake(int type, int cnt) {
  screenShakeType = type; screenShakeCnt = cnt;
}

void moveScreenShake() {
  if ( screenShakeCnt > 0 ) {
    screenShakeCnt--;
  }
}

void drawGLSceneStart() {
  glClear(GL_COLOR_BUFFER_BIT);
  setEyepos();
}

void drawGLSceneEnd() {
  glPopMatrix();
}

void swapGLScene() {
#ifndef HAVE_GLES
  SDL_GL_SwapBuffers();
#else
  EGL_SwapBuffers();
#endif
}


void drawBox(GLfloat x, GLfloat y, GLfloat width, GLfloat height,
	     int r, int g, int b) {
  glPushMatrix();
  glTranslatef(x, y, 0);
  glColor4ub(r, g, b, 128);
#ifndef HAVE_GLES  
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-width, -height,  0);
  glVertex3f( width, -height,  0);
  glVertex3f( width,  height,  0);
  glVertex3f(-width,  height,  0);
  glEnd();
#else
GLfloat vtx1[] = {
	-width, -height,  0,
	width, -height,  0,
	width,  height,  0,
	-width,  height,  0
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx1);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glColor4ub(r, g, b, 255);
#ifndef HAVE_GLES
  glBegin(GL_LINE_LOOP);
  glVertex3f(-width, -height,  0);
  glVertex3f( width, -height,  0);
  glVertex3f( width,  height,  0);
  glVertex3f(-width,  height,  0);
  glEnd();
#else
GLfloat vtx2[] = {
	-width, -height,  0,
	width, -height,  0,
	width,  height,  0,
	-width,  height,  0
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx2);
    glDrawArrays(GL_LINE_LOOP,0,4);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glPopMatrix();
}

void drawLine(GLfloat x1, GLfloat y1, GLfloat z1,
	      GLfloat x2, GLfloat y2, GLfloat z2, int r, int g, int b, int a) {
  glColor4ub(r, g, b, a);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  glVertex3f(x1, y1, z1);
  glVertex3f(x2, y2, z2);
  glEnd();
#else
GLfloat vtx3[] = {
	x1, y1, z1,
	x2, y2, z2
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx3);
    glDrawArrays(GL_LINES,0,2);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void drawLinePart(GLfloat x1, GLfloat y1, GLfloat z1,
		  GLfloat x2, GLfloat y2, GLfloat z2, int r, int g, int b, int a, int len) {
  glColor4ub(r, g, b, a);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  glVertex3f(x1, y1, z1);
  glVertex3f(x1+(x2-x1)*len/256, y1+(y2-y1)*len/256, z1+(z2-z1)*len/256);
  glEnd();
#else
GLfloat vtx4[] = {
	x1, y1, z1,
	x1+(x2-x1)*len/256, y1+(y2-y1)*len/256, z1+(z2-z1)*len/256
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx4);
    glDrawArrays(GL_LINES,0,2);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void drawRollLineAbs(GLfloat x1, GLfloat y1, GLfloat z1,
		     GLfloat x2, GLfloat y2, GLfloat z2, int r, int g, int b, int a, int d1) {
  glPushMatrix();
  glRotatef((float)d1*360/1024, 0, 0, 1);
  glColor4ub(r, g, b, a);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  glVertex3f(x1, y1, z1);
  glVertex3f(x2, y2, z2);
  glEnd();
#else
GLfloat vtx5[] = {
	x1, y1, z1,
	x2, y2, z2
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx5);
    glDrawArrays(GL_LINES,0,2);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glPopMatrix();
}

void drawRollLine(GLfloat x, GLfloat y, GLfloat z, GLfloat width,
		  int r, int g, int b, int a, int d1, int d2) {
  glPushMatrix();
  glTranslatef(x, y, z);
  glRotatef((float)d1*360/1024, 0, 0, 1);
  glRotatef((float)d2*360/1024, 1, 0, 0);
  glColor4ub(r, g, b, a);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  glVertex3f(0, -width, 0);
  glVertex3f(0,  width, 0);
  glEnd();
#else
GLfloat vtx6[] = {
	0, -width, 0,
	0,  width, 0
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx6);
    glDrawArrays(GL_LINES,0,2);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glPopMatrix();
}

void drawSquare(GLfloat x1, GLfloat y1, GLfloat z1,
		GLfloat x2, GLfloat y2, GLfloat z2,
		GLfloat x3, GLfloat y3, GLfloat z3,
		GLfloat x4, GLfloat y4, GLfloat z4,
		int r, int g, int b) {
  glColor4ub(r, g, b, 64);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(x1, y1, z1);
  glVertex3f(x2, y2, z2);
  glVertex3f(x3, y3, z3);
  glVertex3f(x4, y4, z4);
  glEnd();
#else
GLfloat vtx7[] = {
	x1, y1, z1,
	x2, y2, z2,
	x3, y3, z3,
	x4, y4, z4
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx7);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void drawStar(int f, GLfloat x, GLfloat y, GLfloat z, int r, int g, int b, float size) {
  glEnable(GL_TEXTURE_2D);
  if ( f ) {
    glBindTexture(GL_TEXTURE_2D, starTexture);
  } else {
    glBindTexture(GL_TEXTURE_2D, smokeTexture);
  }
  glColor4ub(r, g, b, 255);
  glPushMatrix();
  glTranslatef(x, y, z);
  glRotatef(rand()%360, 0.0f, 0.0f, 1.0f);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(-size, -size,  0);
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f( size, -size,  0);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f( size,  size,  0);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(-size,  size,  0);
  glEnd();
#else
GLfloat vtx8[] = {
	-size, -size,  0,
	size, -size,  0,
	size,  size,  0,
	-size,  size,  0
	};
GLfloat tex1[] = {
	0.0f, 1.0f,
	1.0f, 1.0f,
	1.0f, 0.0f,
	0.0f, 0.0f
	};
      glEnableClientState(GL_VERTEX_ARRAY);
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx8);
      glTexCoordPointer(2, GL_FLOAT, 0, tex1);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
  glPopMatrix();
  glDisable(GL_TEXTURE_2D);
}

#define LASER_ALPHA 100
#define LASER_LINE_ALPHA 50
#define LASER_LINE_ROLL_SPEED 17
#define LASER_LINE_UP_SPEED 16

void drawLaser(GLfloat x, GLfloat y, GLfloat width, GLfloat height,
	       int cc1, int cc2, int cc3, int cc4, int cnt, int type) {
  int i, d;
  float gx, gy;
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  if ( type != 0 ) {
    glColor4ub(cc1, cc1, cc1, LASER_ALPHA);
    glVertex3f(x-width, y, 0);
  }
  glColor4ub(cc2, 255, cc2, LASER_ALPHA);
  glVertex3f(x, y, 0);
  glColor4ub(cc4, 255, cc4, LASER_ALPHA);
  glVertex3f(x, y+height, 0);
  glColor4ub(cc3, cc3, cc3, LASER_ALPHA);
  glVertex3f(x-width, y+height, 0);
  glEnd();
#else
GLfloat vtx9[] = {
	x-width, y, 0,
	x, y, 0,
	x, y+height, 0,
	x-width, y+height, 0
	};
GLfloat vtx10[] = {
	x, y, 0,
	x, y+height, 0,
	x-width, y+height, 0
	};
GLubyte col1[] = {
	cc1, cc1, cc1, LASER_ALPHA,
	cc2, 255, cc2, LASER_ALPHA,
	cc4, 255, cc4, LASER_ALPHA,
	cc3, cc3, cc3, LASER_ALPHA
	};
GLubyte col2[] = {
	cc2, 255, cc2, LASER_ALPHA,
	cc4, 255, cc4, LASER_ALPHA,
	cc3, cc3, cc3, LASER_ALPHA
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
  if ( type != 0 ) {
    glVertexPointer(3, GL_FLOAT, 0, vtx9);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col1);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
	} else {
    glVertexPointer(3, GL_FLOAT, 0, vtx10);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col2); 
    glDrawArrays(GL_TRIANGLE_FAN,0,3);
        }
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  if ( type != 0 ) {
    glColor4ub(cc1, cc1, cc1, LASER_ALPHA);
    glVertex3f(x+width, y, 0);
  }
  glColor4ub(cc2, 255, cc2, LASER_ALPHA);
  glVertex3f(x, y, 0);
  glColor4ub(cc4, 255, cc4, LASER_ALPHA);
  glVertex3f(x, y+height, 0);
  glColor4ub(cc3, cc3, cc3, LASER_ALPHA);
  glVertex3f(x+width, y+height, 0);
  glEnd();
#else
GLfloat vtx11[] = {
	x+width, y, 0,
	x, y, 0,
	x, y+height, 0,
	x+width, y+height, 0
	};
GLfloat vtx12[] = {
	x, y, 0,
	x, y+height, 0,
	x+width, y+height, 0
	};
GLubyte col3[] = {
	cc1, cc1, cc1, LASER_ALPHA,
	cc2, 255, cc2, LASER_ALPHA,
	cc4, 255, cc4, LASER_ALPHA,
	cc3, cc3, cc3, LASER_ALPHA
	};
GLubyte col4[] = {
	cc2, 255, cc2, LASER_ALPHA,
	cc4, 255, cc4, LASER_ALPHA,
	cc3, cc3, cc3, LASER_ALPHA
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
  if ( type != 0 ) {
    glVertexPointer(3, GL_FLOAT, 0, vtx11);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col3);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
	} else {
    glVertexPointer(3, GL_FLOAT, 0, vtx12);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col4); 
    glDrawArrays(GL_TRIANGLE_FAN,0,3);
        } 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  if ( type == 2 ) return;
  glColor4ub(80, 240, 80, LASER_LINE_ALPHA);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  d = (cnt*LASER_LINE_ROLL_SPEED)&(512/4-1);
  for ( i=0 ; i<4 ; i++, d+=(512/4) ) {
    d &= 1023;
    gx = x + width*sctbl[d+256]/256.0f;
    if ( type == 1 ) {
      glVertex3f(gx, y, 0);
    } else {
      glVertex3f(x, y, 0);
    }
    glVertex3f(gx, y+height, 0);
  }
  if ( type == 0 ) {
    glEnd();
    return;
  }
  gy = y + (height/4/LASER_LINE_UP_SPEED) * (cnt&(LASER_LINE_UP_SPEED-1));
  for ( i=0 ; i<4 ; i++, gy+=height/4 ) {
    glVertex3f(x-width, gy, 0);
    glVertex3f(x+width, gy, 0);
  }
  glEnd();
#else
// Code by Pickle

GLfloat vtx[48];
int index = 0;
int count = 0;

#ifndef HAVE_GLES
glBegin(GL_LINES);
#else
index = -1;
#endif

  d = (cnt*LASER_LINE_ROLL_SPEED)&(512/4-1);

  for ( i=0 ; i<4 ; i++, d+=(512/4) ) {

    d &= 1023;

    gx = x + width*sctbl[d+256]/256.0f;

    if ( type == 1 ) {
#ifndef HAVE_GLES
      glVertex3f(gx, y, 0);
#else
      vtx[++index] = gx; vtx[++index] = y; vtx[++index] = 0;
#endif
    } else {
#ifndef HAVE_GLES
      glVertex3f(x, y, 0);
#else
      vtx[++index] = x; vtx[++index] = y; vtx[++index] = 0;
#endif
    }
#ifndef HAVE_GLES
    glVertex3f(gx, y+height, 0);
#else
    vtx[++index] = gx; vtx[++index] = y+height; vtx[++index] = 0;
#endif
  }

  if ( type == 0 ) {
#ifndef HAVE_GLES
    glEnd();

    return;
#else
    count = 8;
  }
  else
  {
    count = 16;

#endif

  gy = y + (height/4/LASER_LINE_UP_SPEED) * (cnt&(LASER_LINE_UP_SPEED-1));

  for ( i=0 ; i<4 ; i++, gy+=height/4 ) {
#ifndef HAVE_GLES
    glVertex3f(x-width, gy, 0);

    glVertex3f(x+width, gy, 0);
#else
    vtx[++index] = x-width; vtx[++index] = gy; vtx[++index] = 0;
    vtx[++index] = x+width; vtx[++index] = gy; vtx[++index] = 0;      
#endif
  }

#ifndef HAVE_GLES
  glEnd();
#else
  } 
  
  glEnableClientState(GL_VERTEX_ARRAY);
 
  glVertexPointer(3, GL_FLOAT, 0, vtx);
  glDrawArrays(GL_LINES,0,count);
 
  glDisableClientState(GL_VERTEX_ARRAY);  
#endif

// End code by Pickle

#endif

}


#define SHAPE_POINT_SIZE 0.05f
#define SHAPE_BASE_COLOR_R 250
#define SHAPE_BASE_COLOR_G 240
#define SHAPE_BASE_COLOR_B 180

#define CORE_HEIGHT 0.2f
#define CORE_RING_SIZE 0.6f

#define SHAPE_POINT_SIZE_L 0.07f

static void drawRing(GLfloat x, GLfloat y, int d1, int d2, int r, int g, int b) {
  int i, d;
  float x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4;
  glPushMatrix();
  glTranslatef(x, y, 0);
  glRotatef((float)d1*360/1024, 0, 0, 1);
  glRotatef((float)d2*360/1024, 1, 0, 0);
  glColor4ub(r, g, b, 255);
  x1 = x2 = 0;
  y1 = y4 =  CORE_HEIGHT/2;
  y2 = y3 = -CORE_HEIGHT/2;
  z1 = z2 = CORE_RING_SIZE;
  for ( i=0,d=0 ; i<8 ; i++ ) {
    d+=(1024/8); d &= 1023;
    x3 = x4 = sctbl[d+256]*CORE_RING_SIZE/256;
    z3 = z4 = sctbl[d]    *CORE_RING_SIZE/256;
    drawSquare(x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4, r, g, b);
    x1 = x3; y1 = y3; z1 = z3;
    x2 = x4; y2 = y4; z2 = z4;
  }
  glPopMatrix();
}

void drawCore(GLfloat x, GLfloat y, int cnt, int r, int g, int b) {
  int i;
  float cy;
  glPushMatrix();
  glTranslatef(x, y, 0);
  glColor4ub(r, g, b, 255);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0);
  glVertex3f( SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0);
  glVertex3f( SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0);
  glVertex3f(-SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0);
  glEnd();
#else
GLfloat vtx19[] = {
	-SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0,
	SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0,
	SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0,
	-SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx19);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glPopMatrix();
  cy = y - CORE_HEIGHT*2.5f;
  for ( i=0 ; i<4 ; i++, cy+=CORE_HEIGHT ) {
    drawRing(x, cy, (cnt*(4+i))&1023, (sctbl[(cnt*(5+i))&1023]/4)&1023, r, g, b);
  }
}

#define SHIP_DRUM_R 0.4f
#define SHIP_DRUM_WIDTH 0.05f
#define SHIP_DRUM_HEIGHT 0.35f

void drawShipShape(GLfloat x, GLfloat y, float d, int inv) {
  int i;
  glPushMatrix();
  glTranslatef(x, y, 0);
  glColor4ub(255, 100, 100, 255);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0);
  glVertex3f( SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0);
  glVertex3f( SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0);
  glVertex3f(-SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0);
  glEnd();
#else
GLfloat vtx20[] = {
	-SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0,
	SHAPE_POINT_SIZE_L, -SHAPE_POINT_SIZE_L,  0,
	SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0,
	-SHAPE_POINT_SIZE_L,  SHAPE_POINT_SIZE_L,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx20);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  if ( inv ) {
    glPopMatrix();
    return;
  }
  glRotatef(d, 0, 1, 0);
    glColor4ub(120, 220, 100, 150);
    /*if ( mode == IKA_MODE ) {
    glColor4ub(180, 200, 160, 150);
  } else {
    glColor4ub(120, 220, 100, 150);
    }*/
  for ( i=0 ; i<8 ; i++ ) {
    glRotatef(45, 0, 1, 0);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-SHIP_DRUM_WIDTH, -SHIP_DRUM_HEIGHT, SHIP_DRUM_R);
    glVertex3f( SHIP_DRUM_WIDTH, -SHIP_DRUM_HEIGHT, SHIP_DRUM_R);
    glVertex3f( SHIP_DRUM_WIDTH,  SHIP_DRUM_HEIGHT, SHIP_DRUM_R);
    glVertex3f(-SHIP_DRUM_WIDTH,  SHIP_DRUM_HEIGHT, SHIP_DRUM_R);
    glEnd();
#else
GLfloat vtx21[] = {
	-SHIP_DRUM_WIDTH, -SHIP_DRUM_HEIGHT, SHIP_DRUM_R,
	SHIP_DRUM_WIDTH, -SHIP_DRUM_HEIGHT, SHIP_DRUM_R,
	SHIP_DRUM_WIDTH,  SHIP_DRUM_HEIGHT, SHIP_DRUM_R,
	-SHIP_DRUM_WIDTH,  SHIP_DRUM_HEIGHT, SHIP_DRUM_R
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx21);
      glDrawArrays(GL_LINE_LOOP,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  }
  glPopMatrix();
}

void drawBomb(GLfloat x, GLfloat y, GLfloat width, int cnt) {
  int i, d, od, c;
  GLfloat x1, y1, x2, y2;
  d = cnt*48; d &= 1023;
  c = 4+(cnt>>3); if ( c > 16 ) c = 16;
  od = 1024/c;
  x1 = (sctbl[d]    *width)/256 + x;
  y1 = (sctbl[d+256]*width)/256 + y;
  for ( i=0 ; i<c ; i++ ) {
    d += od; d &= 1023;
    x2 = (sctbl[d]    *width)/256 + x;
    y2 = (sctbl[d+256]*width)/256 + y;
    drawLine(x1, y1, 0, x2, y2, 0, 255, 255, 255, 255);
    x1 = x2; y1 = y2;
  }
}

void drawCircle(GLfloat x, GLfloat y, GLfloat width, int cnt,
		int r1, int g1, int b1, int r2, int b2, int g2) {
  int i, d;
  GLfloat x1, y1, x2, y2;
#ifndef HAVE_GLES
  if ( (cnt&1) == 0 ) {
    glColor4ub(r1, g1, b1, 64);
  } else {
    glColor4ub(255, 255, 255, 64);
  }
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(x, y, 0);
  d = cnt*48; d &= 1023;
  x1 = (sctbl[d]    *width)/256 + x;
  y1 = (sctbl[d+256]*width)/256 + y;
  glColor4ub(r2, g2, b2, 150);
  for ( i=0 ; i<16 ; i++ ) {
    d += 64; d &= 1023;
    x2 = (sctbl[d]    *width)/256 + x;
    y2 = (sctbl[d+256]*width)/256 + y;
    glVertex3f(x1, y1, 0);
    glVertex3f(x2, y2, 0);
    x1 = x2; y1 = y2;
  }
  glEnd();
#else
GLfloat vtx25[99];
GLubyte col[132];

int index_col = 0;
int index = 0;
index = -1;
index_col = -1;

  if ( (cnt&1) == 0 ) {
      col[++index_col] = r1; col[++index_col] = g1; col[++index_col] = b1; col[++index_col] = b1;
      } else {
      col[++index_col] = 255; col[++index_col] = 255; col[++index_col] = 255; col[++index_col] = 64;
      }

vtx25[++index] = x; vtx25[++index] = y; vtx25[++index] = 0;

  d = cnt*48; d &= 1023;
  x1 = (sctbl[d]    *width)/256 + x;
  y1 = (sctbl[d+256]*width)/256 + y;

  for ( i=0 ; i<16 ; i++ ) {
    d += 64; d &= 1023;
    x2 = (sctbl[d]    *width)/256 + x;
    y2 = (sctbl[d+256]*width)/256 + y;
      vtx25[++index] = x1; vtx25[++index] = y1; vtx25[++index] = 0;
      vtx25[++index] = x2; vtx25[++index] = y2; vtx25[++index] = 0;
      x1 = x2; y1 = y2;
      col[++index_col] = r2; col[++index_col] = g2; col[++index_col] = b2; col[++index_col] = 150;
      col[++index_col] = r2; col[++index_col] = g2; col[++index_col] = b2; col[++index_col] = 150;
    }
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_COLOR_ARRAY);
 
  glVertexPointer(3, GL_FLOAT, 0, vtx25);
  glColorPointer(4, GL_UNSIGNED_BYTE, 0, col);
  glDrawArrays(GL_TRIANGLE_FAN,0,33);
 
  glDisableClientState(GL_VERTEX_ARRAY);
  glDisableClientState(GL_COLOR_ARRAY);  
#endif
}

void drawShape(GLfloat x, GLfloat y, GLfloat size, int d, int cnt, int type,
	       int r, int g, int b) {
  GLfloat sz, sz2;
  sz2 = 0;
  glPushMatrix();
  glTranslatef(x, y, 0);
  glColor4ub(r, g, b, 255);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0);
  glVertex3f( SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0);
  glVertex3f( SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0);
  glVertex3f(-SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0);
  glEnd();
#else
GLfloat vtx26[] = {
	-SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0,
	SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0,
	SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0,
	-SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx26);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  switch ( type ) {
  case 0:
    sz = size/2;
    glRotatef((float)d*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz, -sz,  0);
    glVertex3f( sz, -sz,  0);
    glVertex3f( 0, size,  0);
    glEnd();
#else
GLfloat vtx27[] = {
	-sz, -sz,  0,
	sz, -sz,  0,
	0, size,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx27);
      glDrawArrays(GL_LINE_LOOP,0,3);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 150);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz,  0);
    glVertex3f( sz, -sz,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( 0, size,  0);
    glEnd();
#else
GLfloat vtx28[] = {
	-sz, -sz,  0,
	sz, -sz,  0,
	0, size,  0
	};
GLubyte col5[] = {
	r, g, b, 150,
	r, g, b, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx28);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col5);
    glDrawArrays(GL_TRIANGLE_FAN,0,3);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 1:
    sz = size/2;
    glRotatef((float)((cnt*23)&1023)*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(  0, -size,  0);
    glVertex3f( sz,     0,  0);
    glVertex3f(  0,  size,  0);
    glVertex3f(-sz,     0,  0);
    glEnd();
#else
GLfloat vtx29[] = {
	0, -size,  0,
	sz,     0,  0,
	0,  size,  0,
	-sz,     0,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx29);
      glDrawArrays(GL_LINE_LOOP,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
    glColor4ub(r, g, b, 180);
#ifndef HAVE_GLES
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(  0, -size,  0);
    glVertex3f( sz,     0,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f(  0,  size,  0);
    glVertex3f(-sz,     0,  0);
    glEnd();
#else
GLfloat vtx30[] = {
	0, -size,  0,
	sz,     0,  0,
	0, size,  0,
	-sz,     0,  0
	};
GLubyte col6[] = {
	r, g, b, 180,
	r, g, b, 180,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx30);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col6);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 2:
    sz = size/4; sz2 = size/3*2;
    glRotatef((float)d*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz, -sz2,  0);
    glVertex3f( sz, -sz2,  0);
    glVertex3f( sz,  sz2,  0);
    glVertex3f(-sz,  sz2,  0);
    glEnd();
#else
GLfloat vtx31[] = {
	-sz, -sz2,  0,
	 sz, -sz2,  0,
	 sz,  sz2,  0,
	-sz,  sz2,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx31);
      glDrawArrays(GL_LINE_LOOP,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 120);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz2,  0);
    glVertex3f( sz, -sz2,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( sz, sz2,  0);
    glVertex3f(-sz, sz2,  0);
    glEnd();
#else
GLfloat vtx32[] = {
	-sz, -sz2,  0,
	sz, -sz2,  0,
	sz, sz2,  0,
	-sz, sz2,  0
	};
GLubyte col7[] = {
	r, g, b, 120,
	r, g, b, 120,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx32);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col7);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 3:
    sz = size/2;
    glRotatef((float)((cnt*37)&1023)*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz, -sz,  0);
    glVertex3f( sz, -sz,  0);
    glVertex3f( sz,  sz,  0);
    glVertex3f(-sz,  sz,  0);
    glEnd();
#else
GLfloat vtx33[] = {
	-sz, -sz,  0,
	sz, -sz,  0,
	sz,  sz,  0,
	-sz,  sz,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx33);
      glDrawArrays(GL_LINE_LOOP,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 180);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz,  0);
    glVertex3f( sz, -sz,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( sz,  sz,  0);
    glVertex3f(-sz,  sz,  0);
    glEnd();
#else
GLfloat vtx34[] = {
	-sz, -sz2,  0,
	sz, -sz2,  0,
	sz, sz2,  0,
	-sz, sz2,  0
	};
GLubyte col8[] = {
	r, g, b, 180,
	r, g, b, 180,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx34);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col8);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 4:
    sz = size/2;
    glRotatef((float)((cnt*53)&1023)*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz/2, -sz,  0);
    glVertex3f( sz/2, -sz,  0);
    glVertex3f( sz,  -sz/2,  0);
    glVertex3f( sz,   sz/2,  0);
    glVertex3f( sz/2,  sz,  0);
    glVertex3f(-sz/2,  sz,  0);
    glVertex3f(-sz,   sz/2,  0);
    glVertex3f(-sz,  -sz/2,  0);
    glEnd();
#else
GLfloat vtx35[] = {
	-sz/2, -sz,  0,
	sz/2, -sz,  0,
	sz,  -sz/2,  0,
	sz,   sz/2,  0,
	sz/2,  sz,  0,
	-sz/2,  sz,  0,
	-sz,   sz/2,  0,
	-sz,  -sz/2,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx35);
      glDrawArrays(GL_LINE_LOOP,0,8);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 220);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz/2, -sz,  0);
    glVertex3f( sz/2, -sz,  0);
    glVertex3f( sz,  -sz/2,  0);
    glVertex3f( sz,   sz/2,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( sz/2,  sz,  0);
    glVertex3f(-sz/2,  sz,  0);
    glVertex3f(-sz,   sz/2,  0);
    glVertex3f(-sz,  -sz/2,  0);
    glEnd();
#else
GLfloat vtx36[] = {
	-sz/2, -sz,  0,
	sz/2, -sz,  0,
	sz,  -sz/2,  0,
	sz,   sz/2,  0,
	sz/2,  sz,  0,
	-sz/2,  sz,  0,
	-sz,   sz/2,  0,
	-sz,  -sz/2,  0,
	};
GLubyte col9[] = {
	r, g, b, 220,
	r, g, b, 220,
	r, g, b, 220,
	r, g, b, 220,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx36);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col9);
    glDrawArrays(GL_TRIANGLE_FAN,0,8);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 5:
    sz = size*2/3; sz2 = size/5;
    glRotatef((float)d*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_STRIP);
    glVertex3f(-sz, -sz+sz2,  0);
    glVertex3f( 0, sz+sz2,  0);
    glVertex3f( sz, -sz+sz2,  0);
    glEnd();
#else
GLfloat vtx37[] = {
	-sz, -sz+sz2,  0,
	0, sz+sz2,  0,
	sz, -sz+sz2,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx37);
      glDrawArrays(GL_LINE_STRIP,0,3);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 150);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz+sz2,  0);
    glVertex3f( sz, -sz+sz2,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( 0, sz+sz2,  0);
    glEnd();
#else
GLfloat vtx38[] = {
	-sz, -sz+sz2,  0,
	sz, -sz+sz2,  0,
	0, sz+sz2,  0
	};
GLubyte col10[] = {
	r, g, b, 150,
	r, g, b, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx38);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col10);
    glDrawArrays(GL_TRIANGLE_FAN,0,3);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 6:
    sz = size/2;
    glRotatef((float)((cnt*13)&1023)*360/1024, 0, 0, 1);
    glDisable(GL_BLEND);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz, -sz,  0);
    glVertex3f(  0, -sz,  0);
    glVertex3f( sz,   0,  0);
    glVertex3f( sz,  sz,  0);
    glVertex3f(  0,  sz,  0);
    glVertex3f(-sz,   0,  0);
    glEnd();
#else
GLfloat vtx39[] = {
	-sz, -sz,  0,
	0, -sz,  0,
	sz,   0,  0,
	sz,  sz,  0,
	0,  sz,  0,
	-sz,   0,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx39);
      glDrawArrays(GL_LINE_LOOP,0,6);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(r, g, b, 210);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz,  0);
    glVertex3f(  0, -sz,  0);
    glVertex3f( sz,   0,  0);
    glColor4ub(SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150);
    glVertex3f( sz,  sz,  0);
    glVertex3f(  0,  sz,  0);
    glVertex3f(-sz,   0,  0);
    glEnd();
#else
GLfloat vtx40[] = {
	-sz, -sz,  0,
	0, -sz,  0,
	sz,   0,  0,
	sz,  sz,  0,
	0,  sz,  0,
	-sz,   0,  0
	};
GLubyte col11[] = {
	r, g, b, 210,
	r, g, b, 210,
	r, g, b, 210,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150,
	SHAPE_BASE_COLOR_R, SHAPE_BASE_COLOR_G, SHAPE_BASE_COLOR_B, 150
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx40);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col11);
    glDrawArrays(GL_TRIANGLE_FAN,0,6);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  }
  glPopMatrix();
}

static int ikaClr[2][3][3] = {
  {{230, 230, 255}, {100, 100, 200}, {50, 50, 150}},
  {{0, 0, 0}, {200, 0, 0}, {100, 0, 0}},
};

void drawShapeIka(GLfloat x, GLfloat y, GLfloat size, int d, int cnt, int type, int c) {
  GLfloat sz, sz2, sz3;
  glPushMatrix();
  glTranslatef(x, y, 0);
  glColor4ub(ikaClr[c][0][0], ikaClr[c][0][1], ikaClr[c][0][2], 255);
  glDisable(GL_BLEND);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0);
  glVertex3f( SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0);
  glVertex3f( SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0);
  glVertex3f(-SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0);
  glEnd();
#else
GLfloat vtx41[] = {
	-SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0,
	SHAPE_POINT_SIZE, -SHAPE_POINT_SIZE,  0,
	SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0,
	-SHAPE_POINT_SIZE,  SHAPE_POINT_SIZE,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx41);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glColor4ub(ikaClr[c][0][0], ikaClr[c][0][1], ikaClr[c][0][2], 255);
  switch ( type ) {
  case 0:
    sz = size/2; sz2 = sz/3; sz3 = size*2/3;
    glRotatef((float)d*360/1024, 0, 0, 1);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz, -sz3,  0);
    glVertex3f( sz, -sz3,  0);
    glVertex3f( sz2, sz3,  0);
    glVertex3f(-sz2, sz3,  0);
    glEnd();
#else
GLfloat vtx42[] = {
	-sz, -sz3,  0,
	sz, -sz3,  0,
	sz2, sz3,  0,
	-sz2, sz3,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx42);
      glDrawArrays(GL_LINE_LOOP,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz, -sz3,  0);
    glVertex3f( sz, -sz3,  0);
    glColor4ub(ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250);
    glVertex3f( sz2, sz3,  0);
    glVertex3f(-sz2, sz3,  0);
    glEnd();
#else
GLfloat vtx43[] = {
	-sz, -sz3,  0,
	sz, -sz3,  0,
	sz2, sz3,  0,
	-sz2, sz3,  0
	};
GLubyte col12[] = {
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx43);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col12);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  case 1:
    sz = size/2;
    glRotatef((float)((cnt*53)&1023)*360/1024, 0, 0, 1);
#ifndef HAVE_GLES
    glBegin(GL_LINE_LOOP);
    glVertex3f(-sz/2, -sz,  0);
    glVertex3f( sz/2, -sz,  0);
    glVertex3f( sz,  -sz/2,  0);
    glVertex3f( sz,   sz/2,  0);
    glVertex3f( sz/2,  sz,  0);
    glVertex3f(-sz/2,  sz,  0);
    glVertex3f(-sz,   sz/2,  0);
    glVertex3f(-sz,  -sz/2,  0);
    glEnd();
#else
GLfloat vtx44[] = {
	-sz/2, -sz,  0,
	sz/2, -sz,  0,
	sz,  -sz/2,  0,
	sz,   sz/2,  0,
	sz/2,  sz,  0,
	-sz/2,  sz,  0,
	-sz,   sz/2,  0,
	-sz,  -sz/2,  0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx44);
      glDrawArrays(GL_LINE_LOOP,0,8);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_BLEND);
#ifndef HAVE_GLES
    glColor4ub(ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250);
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(-sz/2, -sz,  0);
    glVertex3f( sz/2, -sz,  0);
    glVertex3f( sz,  -sz/2,  0);
    glVertex3f( sz,   sz/2,  0);
    glColor4ub(ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250);
    glVertex3f( sz/2,  sz,  0);
    glVertex3f(-sz/2,  sz,  0);
    glVertex3f(-sz,   sz/2,  0);
    glVertex3f(-sz,  -sz/2,  0);
    glEnd();
#else
GLfloat vtx45[] = {
	-sz/2, -sz,  0,
	sz/2, -sz,  0,
	sz,  -sz/2,  0,
	sz,   sz/2,  0,
	sz/2,  sz,  0,
	-sz/2,  sz,  0,
	-sz,   sz/2,  0,
	-sz,  -sz/2,  0
	};
GLubyte col13[] = {
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][1][0], ikaClr[c][1][1], ikaClr[c][1][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250,
	ikaClr[c][2][0], ikaClr[c][2][1], ikaClr[c][2][2], 250
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx45);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col13);
    glDrawArrays(GL_TRIANGLE_FAN,0,8);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
    break;
  }
  glPopMatrix();
}

#define SHOT_WIDTH 0.1
#define SHOT_HEIGHT 0.2

static int shtClr[3][3][3] = {
  {{200, 200, 225}, {50, 50, 200}, {200, 200, 225}},
  {{100, 0, 0}, {100, 0, 0}, {200, 0, 0}},
  {{100, 200, 100}, {50, 100, 50}, {100, 200, 100}},
};

void drawShot(GLfloat x, GLfloat y, GLfloat d, int c, float width, float height) {
  glPushMatrix();
  glTranslatef(x, y, 0);
  glRotatef(d, 0, 0, 1);
  glColor4ub(shtClr[c][0][0], shtClr[c][0][1], shtClr[c][0][2], 240);
  glDisable(GL_BLEND);
#ifndef HAVE_GLES
  glBegin(GL_LINES);
  glVertex3f(-width, -height, 0);
  glVertex3f(-width,  height, 0);
  glVertex3f( width, -height, 0);
  glVertex3f( width,  height, 0);
  glEnd();
#else
GLfloat vtx46[] = {
	-width, -height, 0,
	-width,  height, 0,
	width, -height, 0,
	width,  height, 0
	};
      glEnableClientState(GL_VERTEX_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx46);
      glDrawArrays(GL_LINES,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glEnable(GL_BLEND);


#ifndef HAVE_GLES
  glColor4ub(shtClr[c][1][0], shtClr[c][1][1], shtClr[c][1][2], 240);
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(-width, -height, 0);
  glVertex3f( width, -height, 0);
  glColor4ub(shtClr[c][2][0], shtClr[c][2][1], shtClr[c][2][2], 240);
  glVertex3f( width,  height, 0);
  glVertex3f(-width,  height, 0);
  glEnd();
#else
GLfloat vtx47[] = {
	-width, -height, 0,
	width, -height, 0,
	width,  height, 0,
	-width,  height, 0
	};
GLubyte col14[] = {
	shtClr[c][1][0], shtClr[c][1][1], shtClr[c][1][2], 240,
	shtClr[c][1][0], shtClr[c][1][1], shtClr[c][1][2], 240,
	shtClr[c][2][0], shtClr[c][2][1], shtClr[c][2][2], 240,
	shtClr[c][2][0], shtClr[c][2][1], shtClr[c][2][2], 240
	};
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
 
    glVertexPointer(3, GL_FLOAT, 0, vtx47);
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, col14);
    glDrawArrays(GL_TRIANGLE_FAN,0,8);
 
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glPopMatrix();
}

void startDrawBoards() {
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0, 640, 480, 0, -1, 1);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
}

void endDrawBoards() {
  glPopMatrix();
  screenResized();
}

static void drawBoard(int x, int y, int width, int height) {
  glColor4ub(0, 0, 0, 255);
#ifndef HAVE_GLES
  glBegin(GL_QUADS);
  glVertex2f(x,y);
  glVertex2f(x+width,y);
  glVertex2f(x+width,y+height);
  glVertex2f(x,y+height);
  glEnd();
#else
    GLfloat vtx48[] = {
        x,y,
        x+width,y,
        x+width,y+height,
        x,y+height
    };
 
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(2, GL_FLOAT, 0, vtx48);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void drawSideBoards() {
  glDisable(GL_BLEND);
  drawBoard(0, 0, 160, 480);
  drawBoard(480, 0, 160, 480);
  glEnable(GL_BLEND);
  drawScore();
  drawRPanel();
}

void drawTitleBoard() {
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, titleTexture);
  glColor4ub(255, 255, 255, 255);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(350, 78,  0);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f(470, 78,  0);
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(470, 114,  0);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(350, 114,  0);
  glEnd();
#else
GLfloat vtx49[] = {
	350, 78,  0,
	470, 78,  0,
	470, 114,  0,
	350, 114,  0
	};
GLfloat tex3[] = {
	0.0f, 0.0f,
	1.0f, 0.0f,
	1.0f, 1.0f,
	0.0f, 1.0f
	};
      glEnableClientState(GL_VERTEX_ARRAY);
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
 
      glVertexPointer(3, GL_FLOAT, 0, vtx49);
      glTexCoordPointer(2, GL_FLOAT, 0, tex3);
      glDrawArrays(GL_TRIANGLE_FAN,0,4);
 
      glDisableClientState(GL_VERTEX_ARRAY);
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
  glDisable(GL_TEXTURE_2D);
  glColor4ub(200, 200, 200, 255);
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(350, 30, 0);
  glVertex3f(400, 30, 0);
  glVertex3f(380, 56, 0);
  glVertex3f(380, 80, 0);
  glVertex3f(350, 80, 0);
  glEnd();
#else
    GLfloat vtx50[] = {
        350, 30, 0,
        400, 30, 0,
        380, 56, 0,
        380, 80, 0,
	350, 80, 0
    };
 
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx50);
    glDrawArrays(GL_TRIANGLE_FAN,0,5);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
#ifndef HAVE_GLES
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(404, 80, 0);
  glVertex3f(404, 8, 0);
  glVertex3f(440, 8, 0);
  glVertex3f(440, 44, 0);
  glVertex3f(465, 80, 0);
  glEnd();
#else
    GLfloat vtx51[] = {
        404, 80, 0,
        404, 8, 0,
        440, 8, 0,
        440, 44, 0,
	465, 80, 0
    };
 
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx51);
    glDrawArrays(GL_TRIANGLE_FAN,0,5);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
  glColor4ub(255, 255, 255, 255);
#ifndef HAVE_GLES
  glBegin(GL_LINE_LOOP);
  glVertex3f(350, 30, 0);
  glVertex3f(400, 30, 0);
  glVertex3f(380, 56, 0);
  glVertex3f(380, 80, 0);
  glVertex3f(350, 80, 0);
  glEnd();
#else
    GLfloat vtx52[] = {
        350, 30, 0,
        400, 30, 0,
        380, 56, 0,
        380, 80, 0,
	350, 80, 0
    };
 
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx52);
    glDrawArrays(GL_LINE_LOOP,0,5);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
#ifndef HAVE_GLES
  glBegin(GL_LINE_LOOP);
  glVertex3f(404, 80, 0);
  glVertex3f(404, 8, 0);
  glVertex3f(440, 8, 0);
  glVertex3f(440, 44, 0);
  glVertex3f(465, 80, 0);
  glEnd();
#else
    GLfloat vtx53[] = {
        404, 80, 0,
        404, 8, 0,
        440, 8, 0,
        440, 44, 0,
	465, 80, 0
    };
 
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, vtx53);
    glDrawArrays(GL_LINE_LOOP,0,5);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

// Draw the numbers.
int drawNum(int n, int x ,int y, int s, int r, int g, int b) {
  for ( ; ; ) {
    drawLetter(n%10, x, y, s, 3, r, g, b);
    y += s*1.7f;
    n /= 10;
    if ( n <= 0 ) break;
  }
  return y;
}

int drawNumRight(int n, int x ,int y, int s, int r, int g, int b) {
  int d, nd, drawn = 0;
  for ( d = 100000000 ; d > 0 ; d /= 10 ) {
    nd = (int)(n/d);
    if ( nd > 0 || drawn ) {
      n -= d*nd;
      drawLetter(nd%10, x, y, s, 1, r, g, b);
      y += s*1.7f;
      drawn = 1;
    }
  }
  if ( !drawn ) {
    drawLetter(0, x, y, s, 1, r, g, b);
    y += s*1.7f;
  }
  return y;
}

int drawNumCenter(int n, int x ,int y, int s, int r, int g, int b) {
  for ( ; ; ) {
    drawLetter(n%10, x, y, s, 0, r, g, b);
    x -= s*1.7f;
    n /= 10;
    if ( n <= 0 ) break;
  }
  return y;
}

int drawTimeCenter(int n, int x ,int y, int s, int r, int g, int b) {
  int i;
  for ( i=0 ; i<7 ; i++ ) {
    if ( i != 4 ) {
      drawLetter(n%10, x, y, s, 0, r, g, b);
      n /= 10;
    } else {
      drawLetter(n%6, x, y, s, 0, r, g, b);
      n /= 6;
    }
    if ( (i&1) == 1 || i == 0 ) {
      switch ( i ) {
      case 3:
	drawLetter(41, x+s*1.16f, y, s, 0, r, g, b);
	break;
      case 5:
	drawLetter(40, x+s*1.16f, y, s, 0, r, g, b);
	break;
      }
      x -= s*1.7f;
    } else {
      x -= s*2.2f;
    }
    if ( n <= 0 ) break;
  }
  return y;
}

#define JOYSTICK_AXIS 16384

int getPadState() {
  int x = 0, y = 0;
  int pad = 0;
  if ( stick != NULL ) {
    x = SDL_JoystickGetAxis(stick, 0);
    y = SDL_JoystickGetAxis(stick, 1);
  }
  if ( keys[SDLK_RIGHT] == SDL_PRESSED || keys[SDLK_KP6] == SDL_PRESSED || x > JOYSTICK_AXIS ) {
    pad |= PAD_RIGHT;
  }
  if ( keys[SDLK_LEFT] == SDL_PRESSED || keys[SDLK_KP4] == SDL_PRESSED || x < -JOYSTICK_AXIS ) {
    pad |= PAD_LEFT;
  }
  if ( keys[SDLK_DOWN] == SDL_PRESSED || keys[SDLK_KP2] == SDL_PRESSED || y > JOYSTICK_AXIS ) {
    pad |= PAD_DOWN;
  }
  if ( keys[SDLK_UP] == SDL_PRESSED ||  keys[SDLK_KP8] == SDL_PRESSED || y < -JOYSTICK_AXIS ) {
    pad |= PAD_UP;
  }
  return pad;
}

int buttonReversed = 0;

int getButtonState() {
  int btn = 0;
  int btn1 = 0, btn2 = 0, btn3 = 0, btn4 = 0;
  if ( stick != NULL ) {
    btn1 = SDL_JoystickGetButton(stick, 0);
    btn2 = SDL_JoystickGetButton(stick, 1);
    btn3 = SDL_JoystickGetButton(stick, 2);
    btn4 = SDL_JoystickGetButton(stick, 3);
  }
  if ( keys[SDLK_HOME] == SDL_PRESSED || btn1 || btn4 ) {
    if ( !buttonReversed ) {
      btn |= PAD_BUTTON1;
    } else {
      btn |= PAD_BUTTON2;
    }
  }
  if ( keys[SDLK_PAGEDOWN] == SDL_PRESSED || btn2 || btn3 ) {
    if ( !buttonReversed ) {
      btn |= PAD_BUTTON2;
    } else {
      btn |= PAD_BUTTON1;
    }
  }
  return btn;
}
